<?php

namespace App\Http\Controllers;

use App\Models\Purchase;
use App\Models\Sale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

use Spatie\MediaLibrary\MediaCollections\Exceptions\FileCannotBeAdded;

class SaleController extends Controller
{

    public function index()
    {
        $sales = Sale::with('user', 'allocations')->latest()->get();
        return view('sales.index', compact('sales'));
    }


    // نمایش فرم ثبت فروش
    public function create()
    {
        // یافتن خریدهایی که هنوز به طور کامل پوشش داده نشده‌اند
        $purchases = Purchase::whereColumn('total_amount', '>', 'allocated_amount')
            ->orWhereNull('allocated_amount')
            ->get();

        return view('sales.create', compact('purchases'));
    }

    // ثبت فروش و اختصاص آن به خریدها
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'customer_name' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'allocations' => 'required|array',
            'allocations.*.purchase_id' => 'required|exists:purchases,id',
            'allocations.*.allocated_amount' => 'required|numeric|min:0',
        ]);

        $totalAllocatedAmount = collect($validatedData['allocations'])->sum('allocated_amount');
        if ($totalAllocatedAmount > $validatedData['amount']) {
            return back()->withErrors(['amount' => 'مبلغ تخصیص یافته نمی‌تواند بیشتر از مبلغ فروش باشد.'])->withInput();
        }

        DB::beginTransaction();
        try {
            $sale = Sale::create([
                'customer_name' => $validatedData['customer_name'],
                'amount' => $validatedData['amount'],
                'user_id' => Auth::id(),
                'status' => 'در انتظار پرداخت مشتری', // تنظیم وضعیت اولیه
            ]);

            foreach ($validatedData['allocations'] as $allocation) {
                $purchase = Purchase::find($allocation['purchase_id']);
                if ($purchase) {
                    $purchase->allocated_amount += $allocation['allocated_amount'];
                    $purchase->save();
                }

                $sale->allocations()->create([
                    'purchase_id' => $allocation['purchase_id'],
                    'allocated_amount' => $allocation['allocated_amount'],
                ]);
            }

            DB::commit();
            return redirect()->route('sales.index')->with('success', 'فروش با موفقیت ثبت شد.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'خطایی در ثبت فروش رخ داد.']);
        }
    }

    public function finalize(Sale $sale)
    {
        return view('sales.finalize', compact('sale'));
    }

    public function updateStatus(Request $request, Sale $sale)
{
    // اطمینان از وجود فایل در درخواست
    // if (!$request->hasFile('receipt')) {
    //     return back()->with('error', 'فایلی برای آپلود انتخاب نشده است.');
    // }

    try {
        $file = $request->file('receipt');

        // بررسی دستی نوع و حجم فایل
        $allowedMimeTypes = ['image/jpeg', 'image/png', 'application/pdf'];
        $contentType = $file->getClientMimeType();

        if (!in_array($contentType, $allowedMimeTypes)) {
            Log::error("File type not allowed: " . $contentType);
            return back()->with('error', 'نوع فایل پشتیبانی نمی‌شود.');
        }

        if ($file->getSize() > 2048000) { // 2 مگابایت
            Log::error("File size is too large: " . $file->getSize());
            return back()->with('error', 'حجم فایل بیشتر از حد مجاز است.');
        }

        DB::beginTransaction();
        try {
            $fileName = time() . '_' . $file->getClientOriginalName();
            
            // انتقال فایل به پوشه public/uploads
            $file->move(public_path('uploads'), $fileName);
            $receiptPath = 'uploads/' . $fileName;

            // به روز رسانی پایگاه داده
            $sale->update([
                'status' => 'پرداخت شده',
                'receipt_path' => $receiptPath,
            ]);

           // DB::commit();
           // return redirect()->route('sales.index')->with('success', 'فروش با موفقیت نهایی و پرداخت شد.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Database or file move failed: " . $e->getMessage());
            return back()->with('error', 'خطایی در نهایی‌سازی فروش رخ داد.');
        }

    } catch (\Exception $e) {
        Log::error("General file upload error: " . $e->getMessage());
        return back()->with('error', 'خطایی در آپلود فایل رخ داد.');
    }
}
}
