<?php

namespace App\Http\Controllers;

use App\Models\Purchase;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class PurchaseController extends Controller
{
    // نمایش لیست خریدها
    public function index()
    {
        $purchases = Purchase::with(['supplier', 'allocations'])->latest()->get();
        return view('purchases.index', compact('purchases'));
    }
    public function create()
    {
        $suppliers = Supplier::all();
        return view('purchases.create', compact('suppliers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'contact_person' => 'required|string|max:255',
            'total_amount' => 'required|numeric|min:0',
            'payments' => 'required|array|min:1',
            'payments.*.account_holder_name' => 'required|string|max:255',
            'payments.*.card_number' => 'nullable|string|max:255',
            'payments.*.sheba_number' => 'nullable|string|max:255',
            'payments.*.amount' => 'required|numeric|min:0',
        ]);

        $totalPaymentsAmount = collect($request->payments)->sum('amount');
        if ($totalPaymentsAmount != $request->total_amount) {
            throw ValidationException::withMessages([
                'total_amount' => 'مجموع مبالغ واریزی باید با مبلغ کل خرید برابر باشد.'
            ]);
        }

        DB::transaction(function () use ($request) {
            $purchase = Purchase::create([
                'supplier_id' => $request->supplier_id,
                'contact_person' => $request->contact_person,
                'total_amount' => $request->total_amount,
            ]);

            foreach ($request->payments as $payment) {
                $purchase->payments()->create($payment);
            }
        });

        return redirect()->route('purchases.create')->with('status', 'خرید با موفقیت ثبت شد!');
    }
}
